
#include "PolymerSystem.hpp"
std::mutex lock;



Complete_System::Complete_System(const int critical_ring_size,const long initial_particales,const long inactive_particles,const double activation_rate,const double dimerization_rate,const double decay_rate,unsigned seed):
rng_engine(seed),
activation_rate_{activation_rate},
decay_rate_{decay_rate},
dimerization_rate_{dimerization_rate}{
    
    //set up all diagonal elements (single particles) with the initial particles
    _total_number_of_components=0;
    for (int i=0; i<number_of_species; ++i) {
        _all_binding_combinations_of_species[i]=0;
        for (int j=0; j<number_of_species; ++j) {
            _ring_structures[i][j]=0;
            _possible_bindings_2D[i][j]=0;
        }
    }
    
    
    _number_of_inactive_particles=0;
    for (int i=0; i<number_of_species; ++i) {
        _ring_structures[i][i]=initial_particales;
        _total_number_of_components+=initial_particales;
        _total_number_of_components+=inactive_particles;
        _number_of_inactive_particles+=inactive_particles;
    }
    
    //for conveniance the right particle always binds with its left end to the right end of the left one
    for (int i=0; i<number_of_species; ++i) {
        for (int j=1; j<number_of_species; ++j) {
            _possible_bindings_2D[(i+1)%number_of_species][(i+j)%number_of_species]=initial_particales;
        }
    }
    
    for (int i=0; i<number_of_species; ++i) {
        for (int j=0; j<number_of_species; ++j) {
            _all_binding_combinations_of_species[i]+=_possible_bindings_2D[i][j]*_ring_structures[i][j];
        }
    }
    
    //get all reaction parameters (Could be done in the same way as before but memcpy is faster and notation is more compact)
    //memcpy(_inactive_particles.data(), inactive_particles, number_of_species*sizeof(long));
    //memcpy(_reaction_parameters.data(), reaction_parameters, (2*number_of_species+1)*sizeof(double));
    for(int i=0;i<number_of_species;++i){
        _inactive_particles[i]=inactive_particles;
    }
    
    _critical_ring_size=critical_ring_size;
    _undesired_structures=0l;
    _finished_rings=0l;
    _destructable_rings=0l;
    _time=0.0;
    _particles_left_to_activate=true;
    _deactivate_last_part_of_rates=false;
    
    //check for logical errors in the inital data
    if (_critical_ring_size<2) {
        std::cout<<"Critical ring size does not make sense \n";
        exit(EXIT_FAILURE);
    }
    for(auto x:_inactive_particles){
        if(x<0){
            std::cout<<"Initial particle numbers do not make sense\n";
            exit(EXIT_FAILURE);
        }
    }
    for (int i=0; i<number_of_species; ++i) {
        if(_ring_structures[i][i]<0){
            std::cout<<"Initial numbers of species do not make sense\n";
            exit(EXIT_FAILURE);
        }
    }
}
 ////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        Complete_System::Complete_System(Complete_System* other,unsigned seed):
        rng_engine(seed),
        _inactive_particles(other->_inactive_particles),
        _ring_structures(other->_ring_structures),
        _all_binding_combinations_of_species(other->_all_binding_combinations_of_species),
        _possible_bindings_2D(other->_possible_bindings_2D),
        activation_rate_{other->activation_rate_},
        decay_rate_{other->decay_rate_},
        dimerization_rate_{other->dimerization_rate_},
        _rates(other->_rates),
        _particles_left_to_activate(other->_particles_left_to_activate),
        _deactivate_last_part_of_rates(other->_deactivate_last_part_of_rates),
        _rate_counter(other->_rate_counter),
        _critical_ring_size(other->_critical_ring_size),
        _finished_rings(other->_finished_rings),
        _undesired_structures(other->_undesired_structures),
        _destructable_rings(other->_destructable_rings),
        _total_number_of_components(other->_total_number_of_components),
        _number_of_inactive_particles(other->_number_of_inactive_particles),
        _time(other->_time)
        {};


////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// combine two rings to one(reaction)
void Complete_System::combine_rings(const int left_end_1,const int right_end_1, const int left_end_2,const int right_end_2){
    remove_ring(left_end_1, right_end_1);
    remove_ring(left_end_2, right_end_2);
    create_ring(left_end_1, right_end_2);
}
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// account for a new ring after reaction
void Complete_System::create_ring(const int left_end,const int right_end){
    _all_binding_combinations_of_species[left_end]+=_possible_bindings_2D[left_end][right_end];
    ++_ring_structures[left_end][right_end];
    if (right_end<left_end) {
        for (int i=1; i<left_end-right_end; ++i) {
            ++_possible_bindings_2D[right_end+1][right_end+i];
            _all_binding_combinations_of_species[right_end+1]+=_ring_structures[right_end+1][right_end+i];
        }
        if (number_of_species-left_end+right_end+1<_critical_ring_size) {
            ++_destructable_rings;
        }
    }
    else{
        if ((left_end!=right_end)&&(right_end-left_end+1<_critical_ring_size)) {
            ++_destructable_rings;
        }
        int temp=(right_end+1)%number_of_species;
        for (int i=right_end+1; i<number_of_species; ++i) {
            ++_possible_bindings_2D[temp][i];
            _all_binding_combinations_of_species[temp]+=_ring_structures[temp][i];
        }
        for (int i=0; i<left_end; ++i) {
            ++_possible_bindings_2D[temp][i];
            _all_binding_combinations_of_species[temp]+=_ring_structures[temp][i];
        }
    }
}
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// account for removed ring after reaction
void Complete_System::remove_ring(const int left_end,const int right_end){
    _all_binding_combinations_of_species[left_end]-=_possible_bindings_2D[left_end][right_end];
    --_ring_structures[left_end][right_end];
    if (right_end<left_end) {
        if (number_of_species-left_end+right_end+1<_critical_ring_size) {
            --_destructable_rings;
        }
        for (int i=1; i<left_end-right_end; ++i) {
            --_possible_bindings_2D[right_end+1][right_end+i];
            _all_binding_combinations_of_species[right_end+1]-=_ring_structures[right_end+1][right_end+i];
        }
    }
    else{
        int temp=(right_end+1)%number_of_species;
        for (int i=right_end+1; i<number_of_species; ++i) {
            --_possible_bindings_2D[temp][i];
            _all_binding_combinations_of_species[temp]-=_ring_structures[temp][i];
        }
        for (int i=0; i<left_end; ++i) {
            --_possible_bindings_2D[temp][i];
            _all_binding_combinations_of_species[temp]-=_ring_structures[temp][i];
        }
        if ((left_end!=right_end)&&(right_end-left_end+1<_critical_ring_size)) {
            --_destructable_rings;
        }
    }
}
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// decompose a ring into its basic elements
void Complete_System::decompose_ring(const int left_end,const int right_end){
    if (right_end<left_end) {
        for (int i=left_end;i<number_of_species; ++i) {
            create_ring(i, i);
        }
        for (int i=0;i<right_end+1; ++i) {
            create_ring(i, i);
        }
    }
    else{
        for (int i=left_end;i<right_end+1; ++i) {
            create_ring(i, i);
        }
    }
}
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// destroy a single ring(includes decomposing)
void Complete_System::destroy_ring(const int left_end,const int right_end){
    decompose_ring(left_end,right_end);
    remove_ring(left_end, right_end);
}
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//destroy ring event if a to small ring decays
void Complete_System::destroy_small_ring_event(){
    long sum_of_rings=0;
    long ring_index=std::uniform_int_distribution<long>(1,_destructable_rings)(rng_engine);
    int left_end=0;
    for (;left_end<number_of_species; ++left_end) {
        if (left_end+_critical_ring_size-1<number_of_species) {
            for (int i=1;i<_critical_ring_size-1;++i) {
                sum_of_rings+=_ring_structures[left_end][left_end+i];
                if (ring_index<=sum_of_rings) {
                    destroy_ring(left_end, left_end+i);
                    return;
                }
            }
        }
        else{
            for (int i=1;i<_critical_ring_size-1;++i) {
                sum_of_rings+=_ring_structures[left_end][(left_end+i)%number_of_species];
                if (ring_index<=sum_of_rings) {
                    destroy_ring(left_end, (left_end+i)%number_of_species);
                    return;
                }
            }
        }
    }
}
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//determine general event for reactions the int corresponds to left end of reacting component
const int Complete_System::get_first_reacting_component_left_end(){
    int component=0;
    double random_number=std::uniform_real_distribution<double>(0,_rates[_rate_counter])(rng_engine);
    while (_rates[component]<random_number) {
        assert(component<2*number_of_species);
        ++component;
    }
    return component;
}
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//gets right end of ring segment if reaction event takes place
const int Complete_System::get_first_reacting_component_right_end(const int left_end){
    int right_end=0;
    long particle_index=std::uniform_int_distribution<long>(1,_all_binding_combinations_of_species[left_end])(rng_engine);
    long sum_of_binding_partners=_possible_bindings_2D[left_end][right_end]*_ring_structures[left_end][right_end];
    while (sum_of_binding_partners<particle_index) {
        ++right_end;
        assert(right_end<number_of_species);
        sum_of_binding_partners+=_possible_bindings_2D[left_end][right_end]*_ring_structures[left_end][right_end];
    }
    return right_end;
}
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//ger the partner for the reaction
const int Complete_System::get_second_reacting_component_left_end(const int left_end_1,const int right_end_1, const int right_end_2){
    int left_end_2=0;
    long site_index=std::uniform_int_distribution<long>(1,_possible_bindings_2D[left_end_1][right_end_1])(rng_engine);
    long sum_of_binding_partners;
    if (right_end_1<right_end_2) {
        left_end_2=right_end_1+1;
        sum_of_binding_partners=_ring_structures[left_end_2][right_end_2];
        while (sum_of_binding_partners<site_index) {
            ++left_end_2;
            sum_of_binding_partners+=_ring_structures[left_end_2][right_end_2];
        }
    }
    else{
        if (right_end_1+1==number_of_species) {
            left_end_2=0;
        }
        else{
            left_end_2=right_end_1+1;
        }
        sum_of_binding_partners=_ring_structures[left_end_2][right_end_2];
        while (sum_of_binding_partners<site_index) {
            ++left_end_2;
            if (left_end_2==number_of_species) {
                left_end_2=0;
            }
            sum_of_binding_partners+=_ring_structures[left_end_2][right_end_2];
        }
    }
    return left_end_2;
}
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//activate particle event
void Complete_System::activate_particle(const int species){
    --_inactive_particles[species];
    create_ring(species,species);
    --_number_of_inactive_particles;
}
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//get accumulated rates
void Complete_System::make_rates(){
    _rates[0]=_all_binding_combinations_of_species[0];
    ++_rate_counter;
    for (; _rate_counter<number_of_species; ++_rate_counter) {
        _rates[_rate_counter]=_rates[_rate_counter-1]+_all_binding_combinations_of_species[_rate_counter];
    }
    _rates[number_of_species]=_rates[number_of_species-1]+decay_rate_*_destructable_rings;
    if (_particles_left_to_activate) {
        if (_deactivate_last_part_of_rates) {
            _particles_left_to_activate=false;
        }
        for (_rate_counter=number_of_species+1; _rate_counter<2*number_of_species+1; ++_rate_counter) {
            _rates[_rate_counter]=_rates[_rate_counter-1]+_inactive_particles[_rate_counter-number_of_species-1]*activation_rate_;
        }
        --_rate_counter;
        if (_number_of_inactive_particles==0) {
            _deactivate_last_part_of_rates=true;
        }
    }
    
}
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// draw event to happen next
void Complete_System::choose_event(){
    int left_end_binding_chain=get_first_reacting_component_left_end();
    if (left_end_binding_chain==number_of_species) {
        destroy_small_ring_event();
    }
    else if (number_of_species<left_end_binding_chain){
        activate_particle(left_end_binding_chain-number_of_species-1);
    }
    else{
        int right_end_binding_chain=get_first_reacting_component_right_end(left_end_binding_chain);
        int right_end_other_chain;
        if (left_end_binding_chain==0) {
            right_end_other_chain=number_of_species-1;
        }
        else{
            right_end_other_chain=left_end_binding_chain-1;
        }
        int left_end_other_chain=get_second_reacting_component_left_end(left_end_binding_chain, right_end_binding_chain, right_end_other_chain);
        combine_rings(left_end_other_chain, right_end_other_chain, left_end_binding_chain, right_end_binding_chain);
    }
}
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//perform one iteration step
bool Complete_System::iteration_step(){
    _rate_counter=0;
    make_rates();
    if (_rates[_rate_counter]==0) {
        return true;
    }
    _time+=std::exponential_distribution<double>(_rates[_rate_counter])(rng_engine);
    choose_event();
    return false;
}
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//perform complete simulation
void Complete_System::simulate(){
    const unsigned limit=(~0u);
    for (unsigned i=0; i<limit; ++i) {
        if (iteration_step()) {
            break;
        }
    }
}
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//write results to external array
void Complete_System::write_final_results(double* const yield){
    _finished_rings+=_ring_structures[0][number_of_species-1];
    for(int i=1; i<number_of_species;++i){
        _finished_rings+=_ring_structures[i][i-1];
    }
    *yield+=double(_finished_rings)/double(_total_number_of_components/number_of_species);
}

////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
void Complete_System::operator()(const unsigned seed, double* const yield){
    double tmp_yield=0;
    Complete_System sys(this,seed);
    sys.simulate();
    sys.write_final_results(&tmp_yield);
    lock.lock();
    *yield+=tmp_yield;
    lock.unlock();
}
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
void Complete_System::GetEnsAv(const int number_of_ensembles,double* const yield){
    std::vector<std::thread> threads;
    //create different seeds for all threads
    auto random_init=std::bind(std::uniform_int_distribution<unsigned>(0,~(0u)), std::ref(rng_engine));
    //create all threads
    for (int i=0; i<number_of_ensembles; ++i) {
        threads.push_back(std::thread{*this,random_init(), yield});
    }
    //wait for all threads to finish
    for (std::thread& t:threads) {
        if (t.joinable()) {
            t.join();
        }
    }
    
    *yield/=double(number_of_ensembles);
}
